<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Sector;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class SectorTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Storage::fake('public');
    }

    /**
     * Test listing all sectors
     */
    public function test_can_list_all_sectors(): void
    {
        $user = User::factory()->create();

        // Create some sectors
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Healthcare']);
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Finance']);

        // List all sectors (public endpoint)
        $response = $this->getJson('/api/sectors');

        $response->assertStatus(200);
        $data = $response->json();

        $this->assertCount(3, $data);
        $this->assertEquals('Technology', $data[0]['name']);
        $this->assertEquals('Healthcare', $data[1]['name']);
        $this->assertEquals('Finance', $data[2]['name']);
    }

    /**
     * Test creating a sector without image
     */
    public function test_can_create_sector_without_image(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', [
                'name' => 'Technology'
            ]);

        $response->assertStatus(201)
            ->assertJson([
                'message' => 'Sector created successfully',
                'sector' => [
                    'name' => 'Technology',
                    'added_by' => $user->id,
                    'updated_by' => $user->id,
                ]
            ]);

        $this->assertDatabaseHas('sectors', [
            'name' => 'Technology',
            'added_by' => $user->id,
        ]);
    }

    /**
     * Test creating a sector with image
     */
    public function test_can_create_sector_with_image(): void
    {
        $user = User::factory()->create();
        $image = UploadedFile::fake()->image('tech.jpg');

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sectors', [
                'name' => 'Technology',
                'image' => $image,
            ]);

        $response->assertStatus(201)
            ->assertJson([
                'message' => 'Sector created successfully',
            ]);

        // Assert file was stored
        $this->assertTrue(Storage::disk('public')->exists('sectors/' . $image->hashName()));

        // Assert database has the record
        $this->assertDatabaseHas('sectors', [
            'name' => 'Technology',
        ]);

        // Assert response contains full URL
        $data = $response->json('sector');
        $this->assertStringContainsString('/storage/sectors/', $data['image']);
    }

    /**
     * Test creating sector requires authentication
     */
    public function test_creating_sector_requires_authentication(): void
    {
        $response = $this->postJson('/api/sectors', [
            'name' => 'Technology'
        ]);

        $response->assertStatus(401);
    }

    /**
     * Test creating sector requires name
     */
    public function test_creating_sector_requires_name(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', []);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['name']);
    }

    /**
     * Test sector name must be unique
     */
    public function test_sector_name_must_be_unique(): void
    {
        $user = User::factory()->create();

        // Create first sector
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        // Try to create duplicate
        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['name']);
    }

    /**
     * Test image must be valid image file
     */
    public function test_image_must_be_valid_image_file(): void
    {
        $user = User::factory()->create();
        $file = UploadedFile::fake()->create('document.pdf', 100);

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sectors', [
                'name' => 'Technology',
                'image' => $file,
            ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['image']);
    }

    /**
     * Test getting sector by ID
     */
    public function test_can_get_sector_by_id(): void
    {
        $user = User::factory()->create();

        // Create sector
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        $sectorId = $createResponse->json('sector.id');

        // Get sector (public endpoint)
        $response = $this->getJson("/api/sectors/{$sectorId}");

        $response->assertStatus(200)
            ->assertJson([
                'name' => 'Technology',
                'added_by' => $user->id,
            ]);
    }

    /**
     * Test getting non-existent sector returns 404
     */
    public function test_getting_non_existent_sector_returns_404(): void
    {
        $response = $this->getJson('/api/sectors/999');

        $response->assertStatus(404)
            ->assertJson([
                'message' => 'Sector not found'
            ]);
    }

    /**
     * Test getting sector by name
     */
    public function test_can_get_sector_by_name(): void
    {
        $user = User::factory()->create();

        // Create sector
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        // Get sector by name (public endpoint)
        $response = $this->getJson('/api/sectors/name/Technology');

        $response->assertStatus(200)
            ->assertJson([
                'name' => 'Technology',
            ]);
    }

    /**
     * Test updating sector name
     */
    public function test_can_update_sector_name(): void
    {
        $user = User::factory()->create();

        // Create sector
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        $sectorId = $createResponse->json('sector.id');

        // Update sector
        $response = $this->actingAs($user, 'sanctum')
            ->postJson("/api/sectors/{$sectorId}", [
                'name' => 'Advanced Technology'
            ]);

        $response->assertStatus(200)
            ->assertJson([
                'message' => 'Sector updated successfully',
                'sector' => [
                    'name' => 'Advanced Technology',
                ]
            ]);

        $this->assertDatabaseHas('sectors', [
            'id' => $sectorId,
            'name' => 'Advanced Technology',
        ]);
    }

    /**
     * Test updating sector with new image replaces old one
     */
    public function test_updating_sector_with_new_image_replaces_old_one(): void
    {
        $user = User::factory()->create();
        $oldImage = UploadedFile::fake()->image('old.jpg');
        $newImage = UploadedFile::fake()->image('new.jpg');

        // Create sector with image
        $createResponse = $this->actingAs($user, 'sanctum')
            ->post('/api/sectors', [
                'name' => 'Technology',
                'image' => $oldImage,
            ]);

        $sectorId = $createResponse->json('sector.id');
        $oldImagePath = 'sectors/' . $oldImage->hashName();

        $this->assertTrue(Storage::disk('public')->exists($oldImagePath));

        // Update with new image
        $response = $this->actingAs($user, 'sanctum')
            ->post("/api/sectors/{$sectorId}", [
                'image' => $newImage,
            ]);

        $response->assertStatus(200);

        // Assert old image is deleted and new image exists
        $this->assertFalse(Storage::disk('public')->exists($oldImagePath));
        $this->assertTrue(Storage::disk('public')->exists('sectors/' . $newImage->hashName()));
    }

    /**
     * Test updating sector requires authentication
     */
    public function test_updating_sector_requires_authentication(): void
    {
        $user = User::factory()->create();

        // Create sector
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        $sectorId = $createResponse->json('sector.id');

        // Try to update without authentication
        $response = $this->postJson("/api/sectors/{$sectorId}", [
            'name' => 'Updated Technology'
        ]);

        $response->assertStatus(401);
    }

    /**
     * Test deleting sector
     */
    public function test_can_delete_sector(): void
    {
        $user = User::factory()->create();

        // Create sector
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        $sectorId = $createResponse->json('sector.id');

        // Delete sector
        $response = $this->actingAs($user, 'sanctum')
            ->deleteJson("/api/sectors/{$sectorId}");

        $response->assertStatus(200)
            ->assertJson([
                'message' => 'Sector deleted successfully'
            ]);

        // Assert soft delete
        $this->assertSoftDeleted('sectors', [
            'id' => $sectorId,
        ]);
    }

    /**
     * Test deleting sector removes image
     */
    public function test_deleting_sector_removes_image(): void
    {
        $user = User::factory()->create();
        $image = UploadedFile::fake()->image('tech.jpg');

        // Create sector with image
        $createResponse = $this->actingAs($user, 'sanctum')
            ->post('/api/sectors', [
                'name' => 'Technology',
                'image' => $image,
            ]);

        $sectorId = $createResponse->json('sector.id');
        $imagePath = 'sectors/' . $image->hashName();

        $this->assertTrue(Storage::disk('public')->exists($imagePath));

        // Delete sector
        $this->actingAs($user, 'sanctum')
            ->deleteJson("/api/sectors/{$sectorId}");

        // Assert image is deleted
        $this->assertFalse(Storage::disk('public')->exists($imagePath));
    }

    /**
     * Test deleting non-existent sector returns 404
     */
    public function test_deleting_non_existent_sector_returns_404(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->deleteJson('/api/sectors/999');

        $response->assertStatus(404)
            ->assertJson([
                'message' => 'Sector not found'
            ]);
    }

    /**
     * Test deleting sector requires authentication
     */
    public function test_deleting_sector_requires_authentication(): void
    {
        $user = User::factory()->create();

        // Create sector
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        $sectorId = $createResponse->json('sector.id');

        // Try to delete without authentication
        $response = $this->deleteJson("/api/sectors/{$sectorId}");

        $response->assertStatus(401);
    }

    /**
     * Test upsert creates new sector if name doesn't exist
     */
    public function test_upsert_creates_new_sector_if_name_does_not_exist(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sectors/name/Technology', []);

        $response->assertStatus(200)
            ->assertJson([
                'message' => 'Sector saved successfully',
                'sector' => [
                    'name' => 'Technology',
                ]
            ]);

        $this->assertDatabaseHas('sectors', [
            'name' => 'Technology',
        ]);
    }

    /**
     * Test upsert updates existing sector if name exists
     */
    public function test_upsert_updates_existing_sector_if_name_exists(): void
    {
        $user = User::factory()->create();
        $oldImage = UploadedFile::fake()->image('old.jpg');
        $newImage = UploadedFile::fake()->image('new.jpg');

        // Create sector with image
        $this->actingAs($user, 'sanctum')
            ->post('/api/sectors', [
                'name' => 'Technology',
                'image' => $oldImage,
            ]);

        $oldImagePath = 'sectors/' . $oldImage->hashName();
        $this->assertTrue(Storage::disk('public')->exists($oldImagePath));

        // Upsert with new image
        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sectors/name/Technology', [
                'image' => $newImage,
            ]);

        $response->assertStatus(200)
            ->assertJson([
                'message' => 'Sector saved successfully',
            ]);

        // Assert only one sector with this name exists
        $this->assertEquals(1, Sector::where('name', 'Technology')->count());

        // Assert old image is deleted and new image exists
        $this->assertFalse(Storage::disk('public')->exists($oldImagePath));
        $this->assertTrue(Storage::disk('public')->exists('sectors/' . $newImage->hashName()));
    }

    /**
     * Test deleting sector by name
     */
    public function test_can_delete_sector_by_name(): void
    {
        $user = User::factory()->create();

        // Create sector
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        // Delete by name
        $response = $this->actingAs($user, 'sanctum')
            ->deleteJson('/api/sectors/name/Technology');

        $response->assertStatus(200)
            ->assertJson([
                'message' => 'Sector deleted successfully'
            ]);

        $this->assertSoftDeleted('sectors', [
            'name' => 'Technology',
        ]);
    }

    /**
     * Test deleting non-existent sector by name returns 404
     */
    public function test_deleting_non_existent_sector_by_name_returns_404(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->deleteJson('/api/sectors/name/NonExistent');

        $response->assertStatus(404)
            ->assertJson([
                'message' => 'Sector not found'
            ]);
    }

    /**
     * Test sector includes user relationships
     */
    public function test_sector_includes_user_relationships(): void
    {
        $user = User::factory()->create();

        // Create sector
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        $sectorId = $createResponse->json('sector.id');

        // Get sector
        $response = $this->getJson("/api/sectors/{$sectorId}");

        $response->assertStatus(200);

        $data = $response->json();

        // Assert relationships are loaded
        $this->assertArrayHasKey('added_by', $data);
        $this->assertArrayHasKey('updated_by', $data);
        $this->assertEquals($user->id, $data['added_by']['id']);
        $this->assertEquals($user->email, $data['added_by']['email']);
    }

    /**
     * Test image accessor returns full URL
     */
    public function test_image_accessor_returns_full_url(): void
    {
        $user = User::factory()->create();
        $image = UploadedFile::fake()->image('tech.jpg');

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sectors', [
                'name' => 'Technology',
                'image' => $image,
            ]);

        $sector = $response->json('sector');

        // Assert image URL contains storage path
        $this->assertStringContainsString('storage/sectors/', $sector['image']);
        $this->assertStringContainsString($image->hashName(), $sector['image']);
    }

    /**
     * Test sector without image returns null for image field
     */
    public function test_sector_without_image_returns_null(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sectors', ['name' => 'Technology']);

        $sector = $response->json('sector');

        $this->assertNull($sector['image']);
    }
}
