<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Section;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class SectionImageUploadTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Storage::fake('public');
    }

    /**
     * Test single image upload with _image suffix
     */
    public function test_can_upload_single_image_with_image_suffix(): void
    {
        $user = User::factory()->create();
        $file = UploadedFile::fake()->image('hero.jpg');

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sections/home', [
                'title' => 'Home Section',
                'hero_image' => $file,
            ]);

        $response->assertStatus(201);
        $response->assertJson([
            'success' => true,
        ]);

        // Assert file was stored
        $this->assertTrue(Storage::disk('public')->exists('sections/' . $file->hashName()));

        // Assert response contains full URL
        $data = $response->json('data');
        $this->assertArrayHasKey('body', $data);
        $this->assertArrayHasKey('hero_image', $data['body']);
        $this->assertStringContainsString('/storage/sections/', $data['body']['hero_image']);
    }

    /**
     * Test multiple image upload with _images suffix
     */
    public function test_can_upload_multiple_images_with_images_suffix(): void
    {
        $user = User::factory()->create();
        $files = [
            UploadedFile::fake()->image('gallery1.jpg'),
            UploadedFile::fake()->image('gallery2.jpg'),
            UploadedFile::fake()->image('gallery3.jpg'),
        ];

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sections/portfolio', [
                'title' => 'Portfolio Section',
                'hero_images' => $files,
            ]);

        $response->assertStatus(201);
        $response->assertJson([
            'success' => true,
        ]);

        // Assert all files were stored
        foreach ($files as $file) {
            $this->assertTrue(Storage::disk('public')->exists('sections/' . $file->hashName()));
        }

        // Assert response contains full URLs array
        $data = $response->json('data');
        $this->assertArrayHasKey('body', $data);
        $this->assertArrayHasKey('hero_images', $data['body']);
        $this->assertIsArray($data['body']['hero_images']);
        $this->assertCount(3, $data['body']['hero_images']);

        foreach ($data['body']['hero_images'] as $url) {
            $this->assertStringContainsString('/storage/sections/', $url);
        }
    }

    /**
     * Test various image field names ending with _image
     */
    public function test_can_upload_various_image_field_names(): void
    {
        $user = User::factory()->create();
        $heroImage = UploadedFile::fake()->image('hero.jpg');
        $aboutImage = UploadedFile::fake()->image('about.jpg');
        $customImage = UploadedFile::fake()->image('custom.jpg');

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sections/about', [
                'text' => 'About us text',
                'hero_image' => $heroImage,
                'about_image' => $aboutImage,
                'qwiejqwe_image' => $customImage,
            ]);

        $response->assertStatus(201);

        // Assert all files were stored
        $this->assertTrue(Storage::disk('public')->exists('sections/' . $heroImage->hashName()));
        $this->assertTrue(Storage::disk('public')->exists('sections/' . $aboutImage->hashName()));
        $this->assertTrue(Storage::disk('public')->exists('sections/' . $customImage->hashName()));

        // Assert response contains all image URLs
        $data = $response->json('data.body');
        $this->assertArrayHasKey('hero_image', $data);
        $this->assertArrayHasKey('about_image', $data);
        $this->assertArrayHasKey('qwiejqwe_image', $data);
        $this->assertArrayHasKey('text', $data);
        $this->assertEquals('About us text', $data['text']);
    }

    /**
     * Test updating section replaces old images
     */
    public function test_updating_section_replaces_old_images(): void
    {
        $user = User::factory()->create();

        // Create section with initial image
        $oldFile = UploadedFile::fake()->image('old.jpg');
        $this->actingAs($user, 'sanctum')
            ->post('/api/sections/banner', [
                'banner_image' => $oldFile,
            ]);

        $oldFilePath = 'sections/' . $oldFile->hashName();
        $this->assertTrue(Storage::disk('public')->exists($oldFilePath));

        // Update with new image
        $newFile = UploadedFile::fake()->image('new.jpg');
        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sections/banner', [
                'banner_image' => $newFile,
            ]);

        $response->assertStatus(200);

        // Assert old file is deleted and new file exists
        $this->assertFalse(Storage::disk('public')->exists($oldFilePath));
        $this->assertTrue(Storage::disk('public')->exists('sections/' . $newFile->hashName()));
    }

    /**
     * Test deleting section removes all images
     */
    public function test_deleting_section_removes_all_images(): void
    {
        $user = User::factory()->create();

        // Create section with images
        $singleImage = UploadedFile::fake()->image('single.jpg');
        $multipleImages = [
            UploadedFile::fake()->image('multi1.jpg'),
            UploadedFile::fake()->image('multi2.jpg'),
        ];

        $this->actingAs($user, 'sanctum')
            ->post('/api/sections/gallery', [
                'hero_image' => $singleImage,
                'gallery_images' => $multipleImages,
            ]);

        $singlePath = 'sections/' . $singleImage->hashName();
        $multiplePaths = array_map(fn($file) => 'sections/' . $file->hashName(), $multipleImages);

        // Assert files exist
        $this->assertTrue(Storage::disk('public')->exists($singlePath));
        foreach ($multiplePaths as $path) {
            $this->assertTrue(Storage::disk('public')->exists($path));
        }

        // Delete section
        $response = $this->actingAs($user, 'sanctum')
            ->delete('/api/sections/gallery');

        $response->assertStatus(200);

        // Assert all files are deleted
        $this->assertFalse(Storage::disk('public')->exists($singlePath));
        foreach ($multiplePaths as $path) {
            $this->assertFalse(Storage::disk('public')->exists($path));
        }
    }

    /**
     * Test mixed data with images and regular fields
     */
    public function test_mixed_data_with_images_and_regular_fields(): void
    {
        $user = User::factory()->create();
        $image = UploadedFile::fake()->image('test.jpg');

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sections/mixed', [
                'title' => 'Mixed Content',
                'description' => 'This is a description',
                'featured_image' => $image,
                'count' => 10,
                'tags' => ['tag1', 'tag2'],
            ]);

        $response->assertStatus(201);

        $data = $response->json('data.body');

        // Assert regular fields are preserved
        $this->assertEquals('Mixed Content', $data['title']);
        $this->assertEquals('This is a description', $data['description']);
        $this->assertEquals(10, $data['count']);
        $this->assertEquals(['tag1', 'tag2'], $data['tags']);

        // Assert image is uploaded and has full URL
        $this->assertStringContainsString('/storage/sections/', $data['featured_image']);
    }

    /**
     * Test retrieving section returns full image URLs
     */
    public function test_retrieving_section_returns_full_image_urls(): void
    {
        $user = User::factory()->create();
        $image = UploadedFile::fake()->image('retrieve.jpg');

        // Create section
        $this->actingAs($user, 'sanctum')
            ->post('/api/sections/retrieve-test', [
                'test_image' => $image,
            ]);

        // Retrieve section (public endpoint)
        $response = $this->get('/api/sections/retrieve-test');

        $response->assertStatus(200);
        $data = $response->json('data.body');

        $this->assertArrayHasKey('test_image', $data);
        $this->assertStringContainsString('/storage/sections/', $data['test_image']);
        $this->assertStringContainsString($image->hashName(), $data['test_image']);
    }

    /**
     * Test base64 image upload with data URI format
     */
    public function test_can_upload_base64_image_with_data_uri(): void
    {
        $user = User::factory()->create();

        // Create a simple 1x1 red PNG in base64
        $base64Image = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mP8z8DwHwAFBQIAX8jx0gAAAABJRU5ErkJggg==';

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sections/base64-test', [
                'title' => 'Base64 Section',
                'hero_image' => $base64Image,
            ]);

        $response->assertStatus(201);

        // Check that file was stored
        $data = $response->json('data.body');
        $this->assertArrayHasKey('hero_image', $data);
        $this->assertStringContainsString('/storage/sections/', $data['hero_image']);

        // Extract path from URL and verify file exists
        $url = $data['hero_image'];
        $path = str_replace('/storage/', '', parse_url($url, PHP_URL_PATH));
        $this->assertTrue(Storage::disk('public')->exists($path));
    }

    /**
     * Test base64 image upload with raw base64 (without data URI)
     */
    public function test_can_upload_raw_base64_image(): void
    {
        $user = User::factory()->create();

        // Raw base64 for a simple 1x1 red PNG
        $base64Image = 'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mP8z8DwHwAFBQIAX8jx0gAAAABJRU5ErkJggg==';

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sections/raw-base64', [
                'banner_image' => $base64Image,
            ]);

        $response->assertStatus(201);

        $data = $response->json('data.body');
        $this->assertArrayHasKey('banner_image', $data);
        $this->assertStringContainsString('/storage/sections/', $data['banner_image']);

        // Verify file exists
        $url = $data['banner_image'];
        $path = str_replace('/storage/', '', parse_url($url, PHP_URL_PATH));
        $this->assertTrue(Storage::disk('public')->exists($path));
    }

    /**
     * Test multiple base64 images upload
     */
    public function test_can_upload_multiple_base64_images(): void
    {
        $user = User::factory()->create();

        $base64Images = [
            'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mP8z8DwHwAFBQIAX8jx0gAAAABJRU5ErkJggg==',
            'data:image/jpeg;base64,/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQH/2wBDAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQH/wAARCAABAAEDASIAAhEBAxEB/8QAFQABAQAAAAAAAAAAAAAAAAAAAAv/xAAUEAEAAAAAAAAAAAAAAAAAAAAA/8QAFQEBAQAAAAAAAAAAAAAAAAAAAAX/xAAUEQEAAAAAAAAAAAAAAAAAAAAA/9oADAMBAAIRAxEAPwA/8A==',
        ];

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/sections/multiple-base64', [
                'gallery_images' => $base64Images,
            ]);

        $response->assertStatus(201);

        $data = $response->json('data.body');
        $this->assertArrayHasKey('gallery_images', $data);
        $this->assertIsArray($data['gallery_images']);
        $this->assertCount(2, $data['gallery_images']);

        // Verify all files exist
        foreach ($data['gallery_images'] as $url) {
            $this->assertStringContainsString('/storage/sections/', $url);
            $path = str_replace('/storage/', '', parse_url($url, PHP_URL_PATH));
            $this->assertTrue(Storage::disk('public')->exists($path));
        }
    }

    /**
     * Test mixed file upload and base64 upload
     */
    public function test_can_upload_mixed_file_and_base64_images(): void
    {
        $user = User::factory()->create();
        $fileImage = UploadedFile::fake()->image('file.jpg');
        $base64Image = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mP8z8DwHwAFBQIAX8jx0gAAAABJRU5ErkJggg==';

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sections/mixed-upload', [
                'hero_image' => $fileImage,
                'about_image' => $base64Image,
                'title' => 'Mixed Upload Test',
            ]);

        $response->assertStatus(201);

        $data = $response->json('data.body');

        // Check file upload
        $this->assertArrayHasKey('hero_image', $data);
        $this->assertStringContainsString('/storage/sections/', $data['hero_image']);
        $this->assertTrue(Storage::disk('public')->exists('sections/' . $fileImage->hashName()));

        // Check base64 upload
        $this->assertArrayHasKey('about_image', $data);
        $this->assertStringContainsString('/storage/sections/', $data['about_image']);

        // Check regular field
        $this->assertEquals('Mixed Upload Test', $data['title']);
    }

    /**
     * Test array file upload with indexed keys (like curl hero_images[0], hero_images[1])
     */
    public function test_can_upload_array_files_with_indexed_keys(): void
    {
        $user = User::factory()->create();
        $file1 = UploadedFile::fake()->image('image1.jpg');
        $file2 = UploadedFile::fake()->image('image2.jpg');
        $file3 = UploadedFile::fake()->image('image3.jpg');

        // Simulate how curl sends hero_images[0], hero_images[1], hero_images[2]
        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/sections/curl-test', [
                'title' => 'Curl Array Upload',
                'hero_images' => [$file1, $file2, $file3],
            ]);

        $response->assertStatus(201);

        $data = $response->json('data.body');
        $this->assertArrayHasKey('hero_images', $data);
        $this->assertIsArray($data['hero_images']);
        $this->assertCount(3, $data['hero_images']);

        // Verify all files were uploaded
        foreach ([$file1, $file2, $file3] as $file) {
            $this->assertTrue(Storage::disk('public')->exists('sections/' . $file->hashName()));
        }
    }
}

