<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Project;
use App\Models\Client;
use App\Models\Sector;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class ProjectTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Storage::fake('public');
    }

    /**
     * Test listing all projects
     */
    public function test_can_list_all_projects(): void
    {
        $user = User::factory()->create();
        $client = Client::create(['name' => 'Test Client', 'added_by' => $user->id, 'updated_by' => $user->id]);
        $sector = Sector::create(['name' => 'Test Sector', 'added_by' => $user->id, 'updated_by' => $user->id]);

        // Create some projects
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Project 1', 'client_id' => $client->id]);
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Project 2', 'sector_id' => $sector->id]);

        // List all projects (public endpoint)
        $response = $this->getJson('/api/projects');

        $response->assertStatus(200);
        $data = $response->json();

        $this->assertCount(2, $data);
        $this->assertEquals('Project 1', $data[0]['name']);
        $this->assertEquals('Project 2', $data[1]['name']);
    }

    /**
     * Test creating a project with minimal data
     */
    public function test_can_create_project_with_minimal_data(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', [
                'name' => 'Website Redesign'
            ]);

        $response->assertStatus(201)
            ->assertJson([
                'message' => 'Project created successfully',
                'project' => [
                    'name' => 'Website Redesign',
                    'added_by' => $user->id,
                    'updated_by' => $user->id,
                ]
            ]);

        $this->assertDatabaseHas('projects', [
            'name' => 'Website Redesign',
            'added_by' => $user->id,
        ]);
    }

    /**
     * Test creating a project with all fields
     */
    public function test_can_create_project_with_all_fields(): void
    {
        $user = User::factory()->create();
        $client = Client::create(['name' => 'Test Client', 'added_by' => $user->id, 'updated_by' => $user->id]);
        $sector = Sector::create(['name' => 'Technology', 'added_by' => $user->id, 'updated_by' => $user->id]);
        $image = UploadedFile::fake()->image('project.jpg');

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/projects', [
                'name' => 'Website Redesign',
                'description' => 'Complete website redesign project',
                'client_id' => $client->id,
                'sector_id' => $sector->id,
                'image' => $image,
            ]);

        $response->assertStatus(201)
            ->assertJson([
                'message' => 'Project created successfully',
            ]);

        // Assert file was stored
        $this->assertTrue(Storage::disk('public')->exists('projects/' . $image->hashName()));

        // Assert database has the record
        $this->assertDatabaseHas('projects', [
            'name' => 'Website Redesign',
            'description' => 'Complete website redesign project',
            'client_id' => $client->id,
            'sector_id' => $sector->id,
        ]);

        // Assert response contains full URL
        $data = $response->json('project');
        $this->assertStringContainsString('/storage/projects/', $data['image']);
        $this->assertEquals($client->id, $data['client']['id']);
        $this->assertEquals($sector->id, $data['sector']['id']);
    }

    /**
     * Test creating project requires authentication
     */
    public function test_creating_project_requires_authentication(): void
    {
        $response = $this->postJson('/api/projects', [
            'name' => 'Test Project'
        ]);

        $response->assertStatus(401);
    }

    /**
     * Test creating project requires name
     */
    public function test_creating_project_requires_name(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', []);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['name']);
    }

    /**
     * Test client_id must exist
     */
    public function test_client_id_must_exist(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', [
                'name' => 'Test Project',
                'client_id' => 999
            ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['client_id']);
    }

    /**
     * Test sector_id must exist
     */
    public function test_sector_id_must_exist(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', [
                'name' => 'Test Project',
                'sector_id' => 999
            ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['sector_id']);
    }

    /**
     * Test image must be valid image file
     */
    public function test_image_must_be_valid_image_file(): void
    {
        $user = User::factory()->create();
        $file = UploadedFile::fake()->create('document.pdf', 100);

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/projects', [
                'name' => 'Test Project',
                'image' => $file,
            ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['image']);
    }

    /**
     * Test getting project by ID
     */
    public function test_can_get_project_by_id(): void
    {
        $user = User::factory()->create();
        $client = Client::create(['name' => 'Test Client', 'added_by' => $user->id, 'updated_by' => $user->id]);

        // Create project
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', [
                'name' => 'Test Project',
                'client_id' => $client->id
            ]);

        $projectId = $createResponse->json('project.id');

        // Get project (public endpoint)
        $response = $this->getJson("/api/projects/{$projectId}");

        $response->assertStatus(200)
            ->assertJson([
                'name' => 'Test Project',
                'added_by' => $user->id,
            ]);

        // Assert relationships are loaded
        $data = $response->json();
        $this->assertArrayHasKey('client', $data);
        $this->assertEquals($client->id, $data['client']['id']);
    }

    /**
     * Test getting non-existent project returns 404
     */
    public function test_getting_non_existent_project_returns_404(): void
    {
        $response = $this->getJson('/api/projects/999');

        $response->assertStatus(404)
            ->assertJson([
                'message' => 'Project not found'
            ]);
    }

    /**
     * Test getting project by name
     */
    public function test_can_get_project_by_name(): void
    {
        $user = User::factory()->create();

        // Create project
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Website Redesign']);

        // Get project by name (public endpoint)
        $response = $this->getJson('/api/projects/name/Website Redesign');

        $response->assertStatus(200)
            ->assertJson([
                'name' => 'Website Redesign',
            ]);
    }

    /**
     * Test getting projects by client ID
     */
    public function test_can_get_projects_by_client_id(): void
    {
        $user = User::factory()->create();
        $client1 = Client::create(['name' => 'Client 1', 'added_by' => $user->id, 'updated_by' => $user->id]);
        $client2 = Client::create(['name' => 'Client 2', 'added_by' => $user->id, 'updated_by' => $user->id]);

        // Create projects for different clients
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Project 1', 'client_id' => $client1->id]);
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Project 2', 'client_id' => $client1->id]);
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Project 3', 'client_id' => $client2->id]);

        // Get projects for client 1
        $response = $this->getJson("/api/projects/client/{$client1->id}");

        $response->assertStatus(200);
        $data = $response->json();

        $this->assertCount(2, $data);
        $this->assertEquals('Project 1', $data[0]['name']);
        $this->assertEquals('Project 2', $data[1]['name']);
    }

    /**
     * Test getting projects by sector ID
     */
    public function test_can_get_projects_by_sector_id(): void
    {
        $user = User::factory()->create();
        $sector1 = Sector::create(['name' => 'Technology', 'added_by' => $user->id, 'updated_by' => $user->id]);
        $sector2 = Sector::create(['name' => 'Healthcare', 'added_by' => $user->id, 'updated_by' => $user->id]);

        // Create projects for different sectors
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Project A', 'sector_id' => $sector1->id]);
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Project B', 'sector_id' => $sector1->id]);
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Project C', 'sector_id' => $sector2->id]);

        // Get projects for sector 1
        $response = $this->getJson("/api/projects/sector/{$sector1->id}");

        $response->assertStatus(200);
        $data = $response->json();

        $this->assertCount(2, $data);
        $this->assertEquals('Project A', $data[0]['name']);
        $this->assertEquals('Project B', $data[1]['name']);
    }

    /**
     * Test updating project
     */
    public function test_can_update_project(): void
    {
        $user = User::factory()->create();
        $client1 = Client::create(['name' => 'Client 1', 'added_by' => $user->id, 'updated_by' => $user->id]);
        $client2 = Client::create(['name' => 'Client 2', 'added_by' => $user->id, 'updated_by' => $user->id]);

        // Create project
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', [
                'name' => 'Original Name',
                'client_id' => $client1->id
            ]);

        $projectId = $createResponse->json('project.id');

        // Update project
        $response = $this->actingAs($user, 'sanctum')
            ->postJson("/api/projects/{$projectId}", [
                'name' => 'Updated Name',
                'description' => 'New description',
                'client_id' => $client2->id
            ]);

        $response->assertStatus(200)
            ->assertJson([
                'message' => 'Project updated successfully',
                'project' => [
                    'name' => 'Updated Name',
                    'description' => 'New description',
                ]
            ]);

        $this->assertDatabaseHas('projects', [
            'id' => $projectId,
            'name' => 'Updated Name',
            'description' => 'New description',
            'client_id' => $client2->id,
        ]);
    }

    /**
     * Test updating project with new image replaces old one
     */
    public function test_updating_project_with_new_image_replaces_old_one(): void
    {
        $user = User::factory()->create();
        $oldImage = UploadedFile::fake()->image('old.jpg');
        $newImage = UploadedFile::fake()->image('new.jpg');

        // Create project with image
        $createResponse = $this->actingAs($user, 'sanctum')
            ->post('/api/projects', [
                'name' => 'Test Project',
                'image' => $oldImage,
            ]);

        $projectId = $createResponse->json('project.id');
        $oldImagePath = 'projects/' . $oldImage->hashName();

        $this->assertTrue(Storage::disk('public')->exists($oldImagePath));

        // Update with new image
        $response = $this->actingAs($user, 'sanctum')
            ->post("/api/projects/{$projectId}", [
                'image' => $newImage,
            ]);

        $response->assertStatus(200);

        // Assert old image is deleted and new image exists
        $this->assertFalse(Storage::disk('public')->exists($oldImagePath));
        $this->assertTrue(Storage::disk('public')->exists('projects/' . $newImage->hashName()));
    }

    /**
     * Test updating project requires authentication
     */
    public function test_updating_project_requires_authentication(): void
    {
        $user = User::factory()->create();

        // Create project
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Test Project']);

        $projectId = $createResponse->json('project.id');

        // Try to update without authentication
        $response = $this->postJson("/api/projects/{$projectId}", [
            'name' => 'Updated Name'
        ]);

        $response->assertStatus(401);
    }

    /**
     * Test deleting project
     */
    public function test_can_delete_project(): void
    {
        $user = User::factory()->create();

        // Create project
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Test Project']);

        $projectId = $createResponse->json('project.id');

        // Delete project
        $response = $this->actingAs($user, 'sanctum')
            ->deleteJson("/api/projects/{$projectId}");

        $response->assertStatus(200)
            ->assertJson([
                'message' => 'Project deleted successfully'
            ]);

        // Assert soft delete
        $this->assertSoftDeleted('projects', [
            'id' => $projectId,
        ]);
    }

    /**
     * Test deleting project removes image
     */
    public function test_deleting_project_removes_image(): void
    {
        $user = User::factory()->create();
        $image = UploadedFile::fake()->image('project.jpg');

        // Create project with image
        $createResponse = $this->actingAs($user, 'sanctum')
            ->post('/api/projects', [
                'name' => 'Test Project',
                'image' => $image,
            ]);

        $projectId = $createResponse->json('project.id');
        $imagePath = 'projects/' . $image->hashName();

        $this->assertTrue(Storage::disk('public')->exists($imagePath));

        // Delete project
        $this->actingAs($user, 'sanctum')
            ->deleteJson("/api/projects/{$projectId}");

        // Assert image is deleted
        $this->assertFalse(Storage::disk('public')->exists($imagePath));
    }

    /**
     * Test deleting non-existent project returns 404
     */
    public function test_deleting_non_existent_project_returns_404(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->deleteJson('/api/projects/999');

        $response->assertStatus(404)
            ->assertJson([
                'message' => 'Project not found'
            ]);
    }

    /**
     * Test deleting project requires authentication
     */
    public function test_deleting_project_requires_authentication(): void
    {
        $user = User::factory()->create();

        // Create project
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Test Project']);

        $projectId = $createResponse->json('project.id');

        // Try to delete without authentication
        $response = $this->deleteJson("/api/projects/{$projectId}");

        $response->assertStatus(401);
    }

    /**
     * Test upsert creates new project if name doesn't exist
     */
    public function test_upsert_creates_new_project_if_name_does_not_exist(): void
    {
        $user = User::factory()->create();
        $client = Client::create(['name' => 'Test Client', 'added_by' => $user->id, 'updated_by' => $user->id]);

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects/name/New Project', [
                'description' => 'A new project',
                'client_id' => $client->id
            ]);

        $response->assertStatus(200)
            ->assertJson([
                'message' => 'Project saved successfully',
                'project' => [
                    'name' => 'New Project',
                    'description' => 'A new project',
                ]
            ]);

        $this->assertDatabaseHas('projects', [
            'name' => 'New Project',
            'description' => 'A new project',
        ]);
    }

    /**
     * Test upsert updates existing project if name exists
     */
    public function test_upsert_updates_existing_project_if_name_exists(): void
    {
        $user = User::factory()->create();
        $client1 = Client::create(['name' => 'Client 1', 'added_by' => $user->id, 'updated_by' => $user->id]);
        $client2 = Client::create(['name' => 'Client 2', 'added_by' => $user->id, 'updated_by' => $user->id]);
        $oldImage = UploadedFile::fake()->image('old.jpg');
        $newImage = UploadedFile::fake()->image('new.jpg');

        // Create project with image
        $this->actingAs($user, 'sanctum')
            ->post('/api/projects', [
                'name' => 'Existing Project',
                'client_id' => $client1->id,
                'image' => $oldImage,
            ]);

        $oldImagePath = 'projects/' . $oldImage->hashName();
        $this->assertTrue(Storage::disk('public')->exists($oldImagePath));

        // Upsert with new data
        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/projects/name/Existing Project', [
                'description' => 'Updated description',
                'client_id' => $client2->id,
                'image' => $newImage,
            ]);

        $response->assertStatus(200)
            ->assertJson([
                'message' => 'Project saved successfully',
            ]);

        // Assert only one project with this name exists
        $this->assertEquals(1, Project::where('name', 'Existing Project')->count());

        // Assert old image is deleted and new image exists
        $this->assertFalse(Storage::disk('public')->exists($oldImagePath));
        $this->assertTrue(Storage::disk('public')->exists('projects/' . $newImage->hashName()));

        // Assert client was updated
        $project = Project::where('name', 'Existing Project')->first();
        $this->assertEquals($client2->id, $project->client_id);
    }

    /**
     * Test deleting project by name
     */
    public function test_can_delete_project_by_name(): void
    {
        $user = User::factory()->create();

        // Create project
        $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Test Project']);

        // Delete by name
        $response = $this->actingAs($user, 'sanctum')
            ->deleteJson('/api/projects/name/Test Project');

        $response->assertStatus(200)
            ->assertJson([
                'message' => 'Project deleted successfully'
            ]);

        $this->assertSoftDeleted('projects', [
            'name' => 'Test Project',
        ]);
    }

    /**
     * Test project includes all relationships
     */
    public function test_project_includes_all_relationships(): void
    {
        $user = User::factory()->create();
        $client = Client::create(['name' => 'Test Client', 'added_by' => $user->id, 'updated_by' => $user->id]);
        $sector = Sector::create(['name' => 'Technology', 'added_by' => $user->id, 'updated_by' => $user->id]);

        // Create project
        $createResponse = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', [
                'name' => 'Test Project',
                'client_id' => $client->id,
                'sector_id' => $sector->id,
            ]);

        $projectId = $createResponse->json('project.id');

        // Get project
        $response = $this->getJson("/api/projects/{$projectId}");

        $response->assertStatus(200);

        $data = $response->json();

        // Assert all relationships are loaded
        $this->assertArrayHasKey('client', $data);
        $this->assertArrayHasKey('sector', $data);
        $this->assertArrayHasKey('added_by', $data);
        $this->assertArrayHasKey('updated_by', $data);

        $this->assertEquals($client->id, $data['client']['id']);
        $this->assertEquals($sector->id, $data['sector']['id']);
        $this->assertEquals($user->id, $data['added_by']['id']);
    }

    /**
     * Test project without image returns null
     */
    public function test_project_without_image_returns_null(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/projects', ['name' => 'Test Project']);

        $project = $response->json('project');

        $this->assertNull($project['image']);
    }

    /**
     * Test image accessor returns full URL
     */
    public function test_image_accessor_returns_full_url(): void
    {
        $user = User::factory()->create();
        $image = UploadedFile::fake()->image('project.jpg');

        $response = $this->actingAs($user, 'sanctum')
            ->post('/api/projects', [
                'name' => 'Test Project',
                'image' => $image,
            ]);

        $project = $response->json('project');

        // Assert image URL contains storage path
        $this->assertStringContainsString('storage/projects/', $project['image']);
        $this->assertStringContainsString($image->hashName(), $project['image']);
    }
}
