<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class UserService
{
    /**
     * Get all users with their relationships
     */
    public function listAll()
    {
        return User::all();
    }

    /**
     * Find a user by ID
     */
    public function findById(int $id): ?User
    {
        return User::find($id);
    }

    /**
     * Find a user by email
     */
    public function findByEmail(string $email): ?User
    {
        return User::where('email', $email)->first();
    }

    /**
     * Create a new user
     */
    public function create(array $data): User
    {
        $userData = [
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
        ];

        $user = User::create($userData);

        return $user;
    }

    /**
     * Update an existing user
     */
    public function update(int $id, array $data): ?User
    {
        $user = User::find($id);
        if (!$user) {
            return null;
        }

        $updateData = [];

        // Update name if provided
        if (isset($data['name'])) {
            $updateData['name'] = $data['name'];
        }

        // Update email if provided
        if (isset($data['email'])) {
            $updateData['email'] = $data['email'];
        }

        // Update password if provided
        if (isset($data['password'])) {
            $updateData['password'] = Hash::make($data['password']);
        }

        $user->update($updateData);

        return $user;
    }

    /**
     * Delete a user by ID (soft delete)
     */
    public function delete(int $id): bool
    {
        $user = User::find($id);
        if (!$user) {
            return false;
        }

        // Revoke all tokens before deleting
        $user->tokens()->delete();

        $user->delete();
        return true;
    }

    /**
     * Restore a soft-deleted user
     */
    public function restore(int $id): ?User
    {
        $user = User::withTrashed()->find($id);
        if (!$user || !$user->trashed()) {
            return null;
        }

        $user->restore();
        return $user;
    }

    /**
     * Get all trashed (soft-deleted) users
     */
    public function listTrashed()
    {
        return User::onlyTrashed()->get();
    }

    /**
     * Force delete a user (permanent deletion)
     */
    public function forceDelete(int $id): bool
    {
        $user = User::withTrashed()->find($id);
        if (!$user) {
            return false;
        }

        // Revoke all tokens before force deleting
        $user->tokens()->delete();

        $user->forceDelete();
        return true;
    }
}
