<?php

namespace App\Services;

use App\Models\Sector;
use Illuminate\Support\Facades\Auth;

class SectorService
{
    use FlexibleBodyTrait;

    /**
     * Get all sectors with their relationships
     */
    public function listAll()
    {
        return Sector::with(['addedBy', 'updatedBy'])->get();
    }

    /**
     * Find a sector by ID
     */
    public function findById(int $id): ?Sector
    {
        return Sector::with(['addedBy', 'updatedBy'])->find($id);
    }

    /**
     * Find a sector by name
     */
    public function findByName(string $name): ?Sector
    {
        return Sector::where('name', $name)
            ->with(['addedBy', 'updatedBy'])
            ->first();
    }

    /**
     * Create or update a sector by name (upsert)
     */
    public function upsertByName(string $name, array $bodyData, array $files = []): Sector
    {
        $userId = Auth::id();

        // Get existing sector to handle image cleanup
        $existingSector = Sector::where('name', $name)->first();
        $existingBody = $existingSector ? $this->getRawBodyData($existingSector) : null;

        // Process image uploads and merge with body data
        $processedData = $this->processImageUploads($bodyData, $files, $existingBody, 'sectors');

        $sector = Sector::updateOrCreate(
            ['name' => $name],
            [
                'body' => $processedData,
                'updated_by' => $userId,
                'added_by' => $userId,
            ]
        );

        return $sector->load(['addedBy', 'updatedBy']);
    }

    /**
     * Delete a sector by name
     */
    public function deleteByName(string $name): bool
    {
        $sector = Sector::where('name', $name)->first();
        if (!$sector) {
            return false;
        }

        // Delete all images associated with this sector
        $body = $this->getRawBodyData($sector);
        $this->deleteAllImages($body);

        $sector->delete();
        return true;
    }
}
