<?php

namespace App\Services;

use App\Models\Project;
use Illuminate\Support\Facades\Auth;

class ProjectService
{
    use FlexibleBodyTrait;

    /**
     * Get all projects with their relationships
     */
    public function listAll()
    {
        return Project::with(['addedBy', 'updatedBy'])->get();
    }

    /**
     * Find a project by ID
     */
    public function findById(int $id): ?Project
    {
        return Project::with(['addedBy', 'updatedBy'])->find($id);
    }

    /**
     * Find a project by name
     */
    public function findByName(string $name): ?Project
    {
        return Project::where('name', $name)
            ->with(['addedBy', 'updatedBy'])
            ->first();
    }

    /**
     * Create or update a project by name (upsert)
     */
    public function upsertByName(string $name, array $bodyData, array $files = []): Project
    {
        $userId = Auth::id();

        // Get existing project to handle image cleanup
        $existingProject = Project::where('name', $name)->first();
        $existingBody = $existingProject ? $this->getRawBodyData($existingProject) : null;

        // Process image uploads and merge with body data
        $processedData = $this->processImageUploads($bodyData, $files, $existingBody, 'projects');

        $project = Project::updateOrCreate(
            ['name' => $name],
            [
                'body' => $processedData,
                'updated_by' => $userId,
                'added_by' => $userId,
            ]
        );

        return $project->load(['addedBy', 'updatedBy']);
    }

    /**
     * Delete a project by name
     */
    public function deleteByName(string $name): bool
    {
        $project = Project::where('name', $name)->first();
        if (!$project) {
            return false;
        }

        // Delete all images associated with this project
        $body = $this->getRawBodyData($project);
        $this->deleteAllImages($body);

        $project->delete();
        return true;
    }
}
