<?php

namespace App\Services;

use App\Models\Mail;
use App\Models\User;
use App\Models\Client;
use Illuminate\Support\Facades\Mail as MailFacade;
use App\Mail\GenericMail;
use Exception;

class MailService
{
    /**
     * List all mails
     */
    public function listAll()
    {
        return Mail::orderBy('created_at', 'desc')->get();
    }

    /**
     * Find mail by ID
     */
    public function findById(int $id): ?Mail
    {
        return Mail::find($id);
    }

    /**
     * Send email to a user
     */
    public function sendToUser(int $userId, string $subject, string $body): Mail
    {
        $user = User::findOrFail($userId);

        return $this->sendMail(
            $user->name,
            $user->email,
            $subject,
            $body,
            'user',
            $userId
        );
    }

    /**
     * Send email to a client (requires email address to be provided)
     */
    public function sendToClient(int $clientId, string $email, string $subject, string $body): Mail
    {
        $client = Client::findOrFail($clientId);

        return $this->sendMail(
            $client->name,
            $email,
            $subject,
            $body,
            'client',
            $clientId
        );
    }

    /**
     * Send email to a custom email address
     */
    public function sendToEmail(string $name, string $email, string $subject, string $body): Mail
    {
        return $this->sendMail($name, $email, $subject, $body);
    }

    /**
     * Core method to send mail and log it
     */
    private function sendMail(
        string $name,
        string $email,
        string $subject,
        string $body,
        ?string $recipientType = null,
        ?int $recipientId = null
    ): Mail {
        // Create mail record
        $mail = Mail::create([
            'name' => $name,
            'email' => $email,
            'subject' => $subject,
            'body' => $body,
            'status' => 'pending',
            'recipient_type' => $recipientType,
            'recipient_id' => $recipientId,
        ]);

        try {
            // Send the email
            MailFacade::to($email)->send(new GenericMail($name, $subject, $body));

            // Update status to sent
            $mail->update([
                'status' => 'sent',
                'sent_at' => now(),
            ]);
        } catch (Exception $e) {
            // Update status to failed
            $mail->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            throw $e;
        }

        return $mail->fresh();
    }

    /**
     * Resend a failed email
     */
    public function resend(int $id): Mail
    {
        $mail = Mail::findOrFail($id);

        try {
            // Send the email
            MailFacade::to($mail->email)->send(
                new GenericMail($mail->name, $mail->subject, $mail->body)
            );

            // Update status to sent
            $mail->update([
                'status' => 'sent',
                'sent_at' => now(),
                'error_message' => null,
            ]);
        } catch (Exception $e) {
            // Update status to failed
            $mail->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            throw $e;
        }

        return $mail->fresh();
    }

    /**
     * Delete a mail record
     */
    public function delete(int $id): bool
    {
        $mail = Mail::find($id);
        if (!$mail) {
            return false;
        }

        $mail->delete();
        return true;
    }

    /**
     * Get mails by status
     */
    public function getByStatus(string $status)
    {
        return Mail::where('status', $status)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    /**
     * Get mails sent to a specific user
     */
    public function getUserMails(int $userId)
    {
        return Mail::where('recipient_type', 'user')
            ->where('recipient_id', $userId)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    /**
     * Get mails sent to a specific client
     */
    public function getClientMails(int $clientId)
    {
        return Mail::where('recipient_type', 'client')
            ->where('recipient_id', $clientId)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    /**
     * Handle contact form submission
     * - Create mail record in database for the customer
     * - Send notification emails to all users (NOT saved in DB)
     */
    public function handleContactForm(string $name, string $email, string $subject, string $body): array
    {
        $results = [
            'contact_mail' => null,
            'notifications_sent' => 0,
            'failed' => []
        ];

        // 1. Create mail record in database for the customer (NOT sent via email)
        $results['contact_mail'] = Mail::create([
            'name' => $name,
            'email' => $email,
            'subject' => $subject,
            'body' => $body,
            'status' => 'pending',
            'recipient_type' => null,
            'recipient_id' => null,
        ]);

        // 2. Send notification emails to all users (WITHOUT saving to database)
        $users = User::all();

        $notificationBody = "New contact form submission received!\n\n" .
            "From: {$name} ({$email})\n" .
            "Subject: {$subject}\n\n" .
            "Message:\n{$body}\n\n" .
            "Please respond to this inquiry at your earliest convenience.";

        $notificationSubject = "New Contact Form Submission: {$subject}";

        foreach ($users as $user) {
            try {
                // Send email directly without creating DB record
                MailFacade::to($user->email)->send(
                    new GenericMail($user->name, $notificationSubject, $notificationBody)
                );
                $results['notifications_sent']++;
            } catch (Exception $e) {
                $results['failed'][] = [
                    'recipient' => $user->email,
                    'user_id' => $user->id,
                    'error' => $e->getMessage()
                ];
            }
        }

        return $results;
    }
}
