<?php

namespace App\Services;

use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\Eloquent\Model;

trait FlexibleBodyTrait
{
    /**
     * Get raw body data from model without accessor transformation
     */
    private function getRawBodyData(Model $model): array
    {
        $attributes = $model->getAttributes();
        $rawBody = $attributes['body'] ?? null;

        if (is_string($rawBody)) {
            return json_decode($rawBody, true) ?? [];
        }

        return is_array($rawBody) ? $rawBody : [];
    }

    /**
     * Process image uploads and merge with body data
     * Handles both single images (_image) and multiple images (_images)
     * Supports file uploads AND base64 encoded images in body data
     */
    private function processImageUploads(array $bodyData, array $files, ?array $existingBody = null, string $directory = 'uploads'): array
    {
        $processedData = $bodyData;

        // Process file uploads first
        foreach ($files as $key => $value) {
            // Handle single image fields (ending with _image)
            if (str_ends_with($key, '_image') && $value instanceof UploadedFile) {
                // Delete old image if exists
                if ($existingBody && isset($existingBody[$key])) {
                    $this->deleteImage($existingBody[$key]);
                }

                // Store new image
                $path = $value->store($directory, 'public');
                $processedData[$key] = $path;
            }

            // Handle multiple images fields (ending with _images)
            elseif (str_ends_with($key, '_images') && is_array($value)) {
                // Delete old images if exists
                if ($existingBody && isset($existingBody[$key]) && is_array($existingBody[$key])) {
                    foreach ($existingBody[$key] as $oldPath) {
                        $this->deleteImage($oldPath);
                    }
                }

                // Store new images
                $paths = [];
                foreach ($value as $file) {
                    if ($file instanceof UploadedFile) {
                        $paths[] = $file->store($directory, 'public');
                    }
                }
                $processedData[$key] = $paths;
            }
        }

        // Process base64 encoded images from body data
        foreach ($processedData as $key => &$value) {
            // Handle single base64 image (ending with _image)
            if (str_ends_with($key, '_image') && is_string($value) && $this->isBase64Image($value)) {
                // Delete old image if exists
                if ($existingBody && isset($existingBody[$key])) {
                    $this->deleteImage($existingBody[$key]);
                }

                // Store base64 image
                $value = $this->storeBase64Image($value, $directory);
            }

            // Handle multiple base64 images (ending with _images)
            elseif (str_ends_with($key, '_images') && is_array($value)) {
                // Delete old images if exists
                if ($existingBody && isset($existingBody[$key]) && is_array($existingBody[$key])) {
                    foreach ($existingBody[$key] as $oldPath) {
                        $this->deleteImage($oldPath);
                    }
                }

                // Store base64 images
                $paths = [];
                foreach ($value as $item) {
                    if (is_string($item) && $this->isBase64Image($item)) {
                        $paths[] = $this->storeBase64Image($item, $directory);
                    } else {
                        $paths[] = $item; // Keep non-image data as is
                    }
                }
                $value = $paths;
            }
        }

        return $processedData;
    }

    /**
     * Check if a string is a base64 encoded image
     */
    private function isBase64Image(string $string): bool
    {
        // Check for data URI format: data:image/xxx;base64,xxxxx
        if (preg_match('/^data:image\/(\w+);base64,/', $string)) {
            return true;
        }

        // Check for raw base64 (starts with common image file signatures when decoded)
        // Must be long enough to be a meaningful image
        if (strlen($string) > 50 && preg_match('/^[a-zA-Z0-9\/\r\n+]*={0,2}$/', $string)) {
            $decoded = @base64_decode($string, true);
            if ($decoded !== false && strlen($decoded) > 0) {
                // Check for image signatures
                $signatures = [
                    "\xFF\xD8\xFF", // JPEG
                    "\x89\x50\x4E\x47", // PNG (shorter signature for better matching)
                    "GIF87a", // GIF87a
                    "GIF89a", // GIF89a
                    "RIFF", // WEBP (RIFF)
                ];

                foreach ($signatures as $signature) {
                    if (str_starts_with($decoded, $signature)) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    /**
     * Store a base64 encoded image and return the path
     */
    private function storeBase64Image(string $base64String, string $directory): string
    {
        // Extract mime type and base64 data
        if (preg_match('/^data:image\/(\w+);base64,(.+)$/', $base64String, $matches)) {
            $extension = $matches[1];
            $data = base64_decode($matches[2]);
        } else {
            // Raw base64 without data URI
            $data = base64_decode($base64String);

            // Detect extension from image data
            $extension = $this->getImageExtensionFromData($data);
        }

        // Generate unique filename
        $filename = Str::random(40) . '.' . $extension;
        $path = $directory . '/' . $filename;

        // Store the file
        Storage::disk('public')->put($path, $data);

        return $path;
    }

    /**
     * Detect image extension from binary data
     */
    private function getImageExtensionFromData(string $data): string
    {
        if (str_starts_with($data, "\xFF\xD8\xFF")) {
            return 'jpg';
        } elseif (str_starts_with($data, "\x89\x50\x4E\x47\x0D\x0A\x1A\x0A")) {
            return 'png';
        } elseif (str_starts_with($data, "GIF87a") || str_starts_with($data, "GIF89a")) {
            return 'gif';
        } elseif (str_starts_with($data, "\x52\x49\x46\x46")) {
            return 'webp';
        }

        return 'jpg'; // default
    }

    /**
     * Delete a single image from storage
     */
    private function deleteImage(?string $path): void
    {
        if ($path && Storage::disk('public')->exists($path)) {
            Storage::disk('public')->delete($path);
        }
    }

    /**
     * Delete all images from body data
     */
    private function deleteAllImages(array $body): void
    {
        foreach ($body as $key => $value) {
            // Delete single image
            if (str_ends_with($key, '_image') && is_string($value)) {
                $this->deleteImage($value);
            }

            // Delete multiple images
            if (str_ends_with($key, '_images') && is_array($value)) {
                foreach ($value as $path) {
                    if (is_string($path)) {
                        $this->deleteImage($path);
                    }
                }
            }
        }
    }

    /**
     * Flatten nested file arrays from request
     * Converts hero_images[0], hero_images[1] into hero_images => [file1, file2]
     */
    private function flattenFiles(array $files): array
    {
        $flattened = [];

        foreach ($files as $key => $value) {
            if (is_array($value)) {
                // Check if it's an array of UploadedFile objects
                $isFileArray = true;
                foreach ($value as $item) {
                    if (!$item instanceof UploadedFile) {
                        $isFileArray = false;
                        break;
                    }
                }

                if ($isFileArray) {
                    // It's already an array of files, use as is
                    $flattened[$key] = array_values($value);
                } else {
                    // Recursively flatten nested arrays
                    $flattened[$key] = $this->flattenFiles($value);
                }
            } else {
                $flattened[$key] = $value;
            }
        }

        return $flattened;
    }
}

