<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class AuthService
{

    /**
     * Login user
     *
     * @param array $credentials
     * @return array
     */
    public function login(array $credentials): array
    {
        try {
            // Attempt to authenticate user
            if (!Auth::attempt($credentials)) {
                return [
                    'success' => false,
                    'message' => 'Invalid credentials',
                    'data' => null
                ];
            }

            $user = Auth::user();

            // Generate access token (5 minutes)
            $accessTokenResult = $user->createToken('access-token', ['*'], now()->addDays(1));
            $accessToken = str_replace($accessTokenResult->accessToken->id . '|', '', $accessTokenResult->plainTextToken);

            return [
                'success' => true,
                'message' => 'Login successful',
                'data' => [
                    'user' => $user,
                    'access_token' => $accessToken,
                    'token_type' => 'Bearer',
                    'expires_in' => 86400
                ]
            ];
        } catch (\Exception $e) {
            Log::error('Login failed: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => 'Login failed. Please try again.',
                'data' => null
            ];
        }
    }

    /**
     * Logout user
     *
     * @param User $user
     * @return array
     */
    public function logout(User $user): array
    {
        try {
            // Revoke all tokens for the user
            $user->tokens()->delete();

            return [
                'success' => true,
                'message' => 'Logged out successfully',
                'data' => null
            ];
        } catch (\Exception $e) {
            Log::error('Logout failed: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => 'Logout failed. Please try again.',
                'data' => null
            ];
        }
    }

    /**
     * Get authenticated user
     *
     * @return array
     */
    public function getAuthenticatedUser(): array
    {
        try {
            $user = Auth::user();

            if (!$user) {
                return [
                    'success' => false,
                    'message' => 'User not authenticated',
                    'data' => null
                ];
            }

            return [
                'success' => true,
                'message' => 'User retrieved successfully',
                'data' => ['user' => $user]
            ];
        } catch (\Exception $e) {
            Log::error('Get authenticated user failed: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => 'Failed to get user information',
                'data' => null
            ];
        }
    }

}
