<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Support\Facades\Storage;

class Setting extends Model
{
    use SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'body',
        'added_by',
        'updated_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'body' => 'array',
    ];

    /**
     * Get the user that created the model.
     */
    public function addedBy()
    {
        return $this->belongsTo(\App\Models\User::class, 'added_by');
    }

    /**
     * Get the user that last updated the model.
     */
    public function updatedBy()
    {
        return $this->belongsTo(\App\Models\User::class, 'updated_by');
    }

    /**
     * Transform body to include full URLs for image paths
     */
    protected function body(): Attribute
    {
        return Attribute::make(
            get: function ($value) {
                $body = json_decode($value, true) ?? [];

                foreach ($body as $key => &$val) {
                    // Transform single image path to full URL
                    if (str_ends_with($key, '_image') && is_string($val)) {
                        $val = url(Storage::url($val));
                    }

                    // Transform multiple image paths to full URLs
                    if (str_ends_with($key, '_images') && is_array($val)) {
                        $val = array_map(function($path) {
                            return is_string($path) ? url(Storage::url($path)) : $path;
                        }, $val);
                    }
                }

                return $body;
            },
            set: fn ($value) => json_encode($value)
        );
    }
}
