<?php

namespace App\Http\Controllers;

use App\Services\UserService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\Rules\Password;

class UserController extends Controller
{
    protected UserService $userService;

    public function __construct(UserService $userService)
    {
        $this->userService = $userService;
    }

    /**
     * Display a listing of users.
     */
    public function index(): JsonResponse
    {
        $users = $this->userService->listAll();
        return response()->json($users);
    }

    /**
     * Store a newly created user.
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email',
            'password' => ['required', 'string', Password::min(8)],
        ]);

        $user = $this->userService->create($request->only(['name', 'email', 'password']));

        return response()->json([
            'message' => 'User created successfully',
            'user' => $user
        ], 201);
    }

    /**
     * Display the specified user.
     */
    public function show(int $id): JsonResponse
    {
        $user = $this->userService->findById($id);

        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        return response()->json($user);
    }

    /**
     * Update the specified user.
     */
    public function update(Request $request, int $id): JsonResponse
    {
        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'email' => 'sometimes|required|string|email|max:255',
            'password' => ['sometimes', 'required', 'string', Password::min(8)],
        ]);

        $user = $this->userService->update(
            $id,
            $request->only(['name', 'email', 'password'])
        );

        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        return response()->json([
            'message' => 'User updated successfully',
            'user' => $user
        ]);
    }

    /**
     * Remove the specified user (soft delete).
     */
    public function destroy(int $id): JsonResponse
    {
        $deleted = $this->userService->delete($id);

        if (!$deleted) {
            return response()->json(['message' => 'User not found'], 404);
        }

        return response()->json(['message' => 'User deleted successfully']);
    }

    /**
     * Find user by email.
     */
    public function findByEmail(string $email): JsonResponse
    {
        $user = $this->userService->findByEmail($email);

        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        return response()->json($user);
    }

    /**
     * Display a listing of trashed (soft-deleted) users.
     */
    public function trashed(): JsonResponse
    {
        $users = $this->userService->listTrashed();
        return response()->json($users);
    }

    /**
     * Restore a soft-deleted user.
     */
    public function restore(int $id): JsonResponse
    {
        $user = $this->userService->restore($id);

        if (!$user) {
            return response()->json(['message' => 'User not found or not deleted'], 404);
        }

        return response()->json([
            'message' => 'User restored successfully',
            'user' => $user
        ]);
    }

    /**
     * Force delete a user (permanent deletion).
     */
    public function forceDestroy(int $id): JsonResponse
    {
        $deleted = $this->userService->forceDelete($id);

        if (!$deleted) {
            return response()->json(['message' => 'User not found'], 404);
        }

        return response()->json(['message' => 'User permanently deleted']);
    }
}
