<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Services\SettingService;
use Illuminate\Support\Facades\Log;

class SettingController extends Controller
{
    private SettingService $settings;

    public function __construct(SettingService $settings)
    {
        $this->settings = $settings;
    }
    /**
     * Display a listing of all settings.
     *
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        $settings = $this->settings->listAll();

        return response()->json([
            'success' => true,
            'data' => $settings,
        ]);
    }

    /**
     * Display the specified setting by name.
     *
     * @param string $name
     * @return JsonResponse
     */
    public function show(string $name): JsonResponse
    {
        $setting = $this->settings->findByName($name);

        if (!$setting) {
            return response()->json([
                'success' => false,
                'message' => "Setting '{$name}' not found",
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $setting,
        ]);
    }

    /**
     * Create or update a setting by name.
     * Accepts any data structure in the request body including file uploads.
     *
     * Handles image uploads automatically:
     * - Single images: fields ending with _image (e.g., logo_image, favicon_image)
     * - Multiple images: fields ending with _images (e.g., social_images, partner_images)
     *
     * @param Request $request
     * @param string $name
     * @return JsonResponse
     */
    public function upsert(Request $request, string $name): JsonResponse
    {
        // Get all request data except files
        $bodyData = $request->except(array_keys($request->allFiles()));

        // Get all file uploads and flatten nested arrays
        $files = $this->flattenFiles($request->allFiles());

        // Debug logging
        Log::info('Setting Upload Debug', [
            'setting_name' => $name,
            'bodyData' => $bodyData,
            'raw_files' => array_keys($request->allFiles()),
            'flattened_files' => array_map(function($file) {
                if (is_array($file)) {
                    return array_map(fn($f) => $f instanceof \Illuminate\Http\UploadedFile ? $f->getClientOriginalName() : 'not-a-file', $file);
                }
                return $file instanceof \Illuminate\Http\UploadedFile ? $file->getClientOriginalName() : 'not-a-file';
            }, $files),
        ]);

        // Get the  user ID if available
        $existing = $this->settings->findByName($name);
        $setting = $this->settings->upsertByName($name, $bodyData, $files);

        return response()->json([
            'success' => true,
            'message' => $existing ? "Setting '{$name}' updated successfully" : "Setting '{$name}' created successfully",
            'data' => $setting,
        ], $existing ? 200 : 201);
    }

    /**
     * Flatten nested file arrays from request
     * Converts logo_images[0], logo_images[1] into logo_images => [file1, file2]
     */
    private function flattenFiles(array $files): array
    {
        $flattened = [];

        foreach ($files as $key => $value) {
            if (is_array($value)) {
                // Check if it's an array of UploadedFile objects
                $isFileArray = true;
                foreach ($value as $item) {
                    if (!$item instanceof \Illuminate\Http\UploadedFile) {
                        $isFileArray = false;
                        break;
                    }
                }

                if ($isFileArray) {
                    // It's already an array of files, use as is
                    $flattened[$key] = array_values($value);
                } else {
                    // Recursively flatten nested arrays
                    $flattened[$key] = $this->flattenFiles($value);
                }
            } else {
                $flattened[$key] = $value;
            }
        }

        return $flattened;
    }

    /**
     * Remove the specified setting by name.
     *
     * @param string $name
     * @return JsonResponse
     */
    public function destroy(string $name): JsonResponse
    {
        $deleted = $this->settings->deleteByName($name);
        if (!$deleted) {
            return response()->json([
                'success' => false,
                'message' => "Setting '{$name}' not found",
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => "Setting '{$name}' deleted successfully",
        ]);
    }
}
