<?php

namespace App\Http\Controllers;

use App\Models\Section;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Services\SectionService;
use Illuminate\Support\Facades\Log;

class SectionController extends Controller
{
    private SectionService $sections;

    public function __construct(SectionService $sections)
    {
        $this->sections = $sections;
    }
    /**
     * Display a listing of all sections.
     *
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        $sections = $this->sections->listAll();

        return response()->json([
            'success' => true,
            'data' => $sections,
        ]);
    }

    /**
     * Display the specified section by name.
     *
     * @param string $name
     * @return JsonResponse
     */
    public function show(string $name): JsonResponse
    {
        $section = $this->sections->findByName($name);

        if (!$section) {
            return response()->json([
                'success' => false,
                'message' => "Section '{$name}' not found",
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $section,
        ]);
    }

    /**
     * Create or update a section by name.
     * Accepts any data structure in the request body including file uploads.
     *
     * Handles image uploads automatically:
     * - Single images: fields ending with _image (e.g., hero_image, about_image)
     * - Multiple images: fields ending with _images (e.g., hero_images, gallery_images)
     *
     * @param Request $request
     * @param string $name
     * @return JsonResponse
     */
    public function upsert(Request $request, string $name): JsonResponse
    {
        // Get all request data except files
        $bodyData = $request->except(array_keys($request->allFiles()));

        // Get all file uploads and flatten nested arrays
        $files = $this->flattenFiles($request->allFiles());

        // Debug logging
        Log::info('Section Upload Debug', [
            'section_name' => $name,
            'bodyData' => $bodyData,
            'raw_files' => array_keys($request->allFiles()),
            'flattened_files' => array_map(function($file) {
                if (is_array($file)) {
                    return array_map(fn($f) => $f instanceof \Illuminate\Http\UploadedFile ? $f->getClientOriginalName() : 'not-a-file', $file);
                }
                return $file instanceof \Illuminate\Http\UploadedFile ? $file->getClientOriginalName() : 'not-a-file';
            }, $files),
        ]);

        // Get the  user ID if available
        $existing = $this->sections->findByName($name);
        $section = $this->sections->upsertByName($name, $bodyData, $files);

        return response()->json([
            'success' => true,
            'message' => $existing ? "Section '{$name}' updated successfully" : "Section '{$name}' created successfully",
            'data' => $section,
        ], $existing ? 200 : 201);
    }

    /**
     * Flatten nested file arrays from request
     * Converts hero_images[0], hero_images[1] into hero_images => [file1, file2]
     */
    private function flattenFiles(array $files): array
    {
        $flattened = [];

        foreach ($files as $key => $value) {
            if (is_array($value)) {
                // Check if it's an array of UploadedFile objects
                $isFileArray = true;
                foreach ($value as $item) {
                    if (!$item instanceof \Illuminate\Http\UploadedFile) {
                        $isFileArray = false;
                        break;
                    }
                }

                if ($isFileArray) {
                    // It's already an array of files, use as is
                    $flattened[$key] = array_values($value);
                } else {
                    // Recursively flatten nested arrays
                    $flattened[$key] = $this->flattenFiles($value);
                }
            } else {
                $flattened[$key] = $value;
            }
        }

        return $flattened;
    }

    /**
     * Remove the specified section by name.
     *
     * @param string $name
     * @return JsonResponse
     */
    public function destroy(string $name): JsonResponse
    {
        $deleted = $this->sections->deleteByName($name);
        if (!$deleted) {
            return response()->json([
                'success' => false,
                'message' => "Section '{$name}' not found",
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => "Section '{$name}' deleted successfully",
        ]);
    }
}
