<?php

namespace App\Http\Controllers;

use App\Services\ProjectService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class ProjectController extends Controller
{
    private ProjectService $projects;

    public function __construct(ProjectService $projects)
    {
        $this->projects = $projects;
    }

    /**
     * Display a listing of all projects.
     */
    public function index(): JsonResponse
    {
        $projects = $this->projects->listAll();

        return response()->json([
            'success' => true,
            'data' => $projects,
        ]);
    }

    /**
     * Display the specified project by name.
     */
    public function show(string $name): JsonResponse
    {
        $project = $this->projects->findByName($name);

        if (!$project) {
            return response()->json([
                'success' => false,
                'message' => "Project '{$name}' not found",
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $project,
        ]);
    }

    /**
     * Create or update a project by name.
     * Accepts any data structure in the request body including file uploads.
     */
    public function upsert(Request $request, string $name): JsonResponse
    {
        // Get all request data except files
        $bodyData = $request->except(array_keys($request->allFiles()));

        // Get all file uploads and flatten nested arrays
        $files = $this->flattenFiles($request->allFiles());

        // Get the existing project if it exists
        $existing = $this->projects->findByName($name);
        $project = $this->projects->upsertByName($name, $bodyData, $files);

        return response()->json([
            'success' => true,
            'message' => $existing ? "Project '{$name}' updated successfully" : "Project '{$name}' created successfully",
            'data' => $project,
        ], $existing ? 200 : 201);
    }

    /**
     * Remove the specified project by name.
     */
    public function destroy(string $name): JsonResponse
    {
        $deleted = $this->projects->deleteByName($name);
        if (!$deleted) {
            return response()->json([
                'success' => false,
                'message' => "Project '{$name}' not found",
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => "Project '{$name}' deleted successfully",
        ]);
    }

    /**
     * Flatten nested file arrays from request
     * Converts hero_images[0], hero_images[1] into hero_images => [file1, file2]
     */
    private function flattenFiles(array $files): array
    {
        $flattened = [];

        foreach ($files as $key => $value) {
            if (is_array($value)) {
                // Check if it's an array of UploadedFile objects
                $isFileArray = true;
                foreach ($value as $item) {
                    if (!$item instanceof \Illuminate\Http\UploadedFile) {
                        $isFileArray = false;
                        break;
                    }
                }

                if ($isFileArray) {
                    // It's already an array of files, use as is
                    $flattened[$key] = array_values($value);
                } else {
                    // Recursively flatten nested arrays
                    $flattened[$key] = $this->flattenFiles($value);
                }
            } else {
                $flattened[$key] = $value;
            }
        }

        return $flattened;
    }
}
