<?php

namespace App\Http\Controllers;

use App\Services\MailService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class MailController extends Controller
{
    protected MailService $mailService;

    public function __construct(MailService $mailService)
    {
        $this->mailService = $mailService;
    }

    /**
     * Display a listing of all mails.
     */
    public function index(): JsonResponse
    {
        $mails = $this->mailService->listAll();
        return response()->json($mails);
    }

    /**
     * Display the specified mail.
     */
    public function show(int $id): JsonResponse
    {
        $mail = $this->mailService->findById($id);

        if (!$mail) {
            return response()->json(['message' => 'Mail not found'], 404);
        }

        return response()->json($mail);
    }

    /**
     * Send email to a user.
     */
    public function sendToUser(Request $request): JsonResponse
    {
        $request->validate([
            'user_id' => 'required|integer|exists:users,id',
            'subject' => 'required|string|max:255',
            'body' => 'required|string',
        ]);

        try {
            $mail = $this->mailService->sendToUser(
                $request->user_id,
                $request->subject,
                $request->body
            );

            return response()->json([
                'message' => 'Email sent successfully to user',
                'mail' => $mail
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to send email',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Send email to a client.
     */
    public function sendToClient(Request $request): JsonResponse
    {
        $request->validate([
            'client_id' => 'required|integer|exists:clients,id',
            'email' => 'required|email',
            'subject' => 'required|string|max:255',
            'body' => 'required|string',
        ]);

        try {
            $mail = $this->mailService->sendToClient(
                $request->client_id,
                $request->email,
                $request->subject,
                $request->body
            );

            return response()->json([
                'message' => 'Email sent successfully to client',
                'mail' => $mail
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to send email',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Send email to a custom email address.
     */
    public function sendToEmail(Request $request): JsonResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'subject' => 'required|string|max:255',
            'body' => 'required|string',
        ]);

        try {
            $mail = $this->mailService->sendToEmail(
                $request->name,
                $request->email,
                $request->subject,
                $request->body
            );

            return response()->json([
                'message' => 'Email sent successfully',
                'mail' => $mail
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to send email',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Resend a failed email.
     */
    public function resend(int $id): JsonResponse
    {
        try {
            $mail = $this->mailService->resend($id);

            return response()->json([
                'message' => 'Email resent successfully',
                'mail' => $mail
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to resend email',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get mails by status.
     */
    public function getByStatus(string $status): JsonResponse
    {
        $mails = $this->mailService->getByStatus($status);
        return response()->json($mails);
    }

    /**
     * Get all mails sent to a specific user.
     */
    public function getUserMails(int $userId): JsonResponse
    {
        $mails = $this->mailService->getUserMails($userId);
        return response()->json($mails);
    }

    /**
     * Get all mails sent to a specific client.
     */
    public function getClientMails(int $clientId): JsonResponse
    {
        $mails = $this->mailService->getClientMails($clientId);
        return response()->json($mails);
    }

    /**
     * Remove the specified mail.
     */
    public function destroy(int $id): JsonResponse
    {
        $deleted = $this->mailService->delete($id);

        if (!$deleted) {
            return response()->json(['message' => 'Mail not found'], 404);
        }

        return response()->json(['message' => 'Mail deleted successfully']);
    }

    /**
     * Handle contact form submission.
     * Creates mail record in database and notifies all users.
     */
    public function contactForm(Request $request): JsonResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'subject' => 'required|string|max:255',
            'body' => 'required|string',
        ]);

        try {
            $results = $this->mailService->handleContactForm(
                $request->name,
                $request->email,
                $request->subject,
                $request->body
            );

            $response = [
                'message' => 'Contact form submitted successfully',
                'contact_mail_id' => $results['contact_mail']->id,
                'users_notified' => $results['notifications_sent'],
                'failed_count' => count($results['failed']),
            ];

            // Include details if there were any failures
            if (count($results['failed']) > 0) {
                $response['failures'] = $results['failed'];
            }

            return response()->json($response, 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to process contact form',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
