<?php

namespace App\Http\Controllers;

use App\Services\ClientService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class ClientController extends Controller
{
    private ClientService $clients;

    public function __construct(ClientService $clients)
    {
        $this->clients = $clients;
    }

    /**
     * Display a listing of all clients.
     */
    public function index(): JsonResponse
    {
        $clients = $this->clients->listAll();

        return response()->json([
            'success' => true,
            'data' => $clients,
        ]);
    }

    /**
     * Display the specified client by name.
     */
    public function show(string $name): JsonResponse
    {
        $client = $this->clients->findByName($name);

        if (!$client) {
            return response()->json([
                'success' => false,
                'message' => "Client '{$name}' not found",
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $client,
        ]);
    }

    /**
     * Create or update a client by name.
     * Accepts any data structure in the request body including file uploads.
     */
    public function upsert(Request $request, string $name): JsonResponse
    {
        // Get all request data except files
        $bodyData = $request->except(array_keys($request->allFiles()));

        // Get all file uploads and flatten nested arrays
        $files = $this->flattenFiles($request->allFiles());

        // Get the existing client if it exists
        $existing = $this->clients->findByName($name);
        $client = $this->clients->upsertByName($name, $bodyData, $files);

        return response()->json([
            'success' => true,
            'message' => $existing ? "Client '{$name}' updated successfully" : "Client '{$name}' created successfully",
            'data' => $client,
        ], $existing ? 200 : 201);
    }

    /**
     * Remove the specified client by name.
     */
    public function destroy(string $name): JsonResponse
    {
        $deleted = $this->clients->deleteByName($name);
        if (!$deleted) {
            return response()->json([
                'success' => false,
                'message' => "Client '{$name}' not found",
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => "Client '{$name}' deleted successfully",
        ]);
    }

    /**
     * Flatten nested file arrays from request
     * Converts hero_images[0], hero_images[1] into hero_images => [file1, file2]
     */
    private function flattenFiles(array $files): array
    {
        $flattened = [];

        foreach ($files as $key => $value) {
            if (is_array($value)) {
                // Check if it's an array of UploadedFile objects
                $isFileArray = true;
                foreach ($value as $item) {
                    if (!$item instanceof \Illuminate\Http\UploadedFile) {
                        $isFileArray = false;
                        break;
                    }
                }

                if ($isFileArray) {
                    // It's already an array of files, use as is
                    $flattened[$key] = array_values($value);
                } else {
                    // Recursively flatten nested arrays
                    $flattened[$key] = $this->flattenFiles($value);
                }
            } else {
                $flattened[$key] = $value;
            }
        }

        return $flattened;
    }
}
