# Section Image Upload Implementation

## Overview
This implementation allows flexible image uploads for sections, supporting both file uploads and base64 encoded images in the request body.

## Features

### 1. **Flexible Field Names**
Any field ending with `_image` (singular) or `_images` (plural) will be automatically processed:
- `hero_image` ✅
- `about_image` ✅
- `qwiejqwe_image` ✅
- `banner_image` ✅
- etc.

### 2. **Multiple Upload Methods**

#### File Upload (Multipart Form Data)
```bash
# Single image
POST /api/sections/home
Content-Type: multipart/form-data

{
  "title": "Home Section",
  "hero_image": [file]
}

# Multiple images
POST /api/sections/gallery
Content-Type: multipart/form-data

{
  "title": "Gallery",
  "hero_images": [file1, file2, file3]
}
```

#### Base64 Encoded Images (JSON)
```bash
# Data URI format
POST /api/sections/about
Content-Type: application/json

{
  "title": "About",
  "hero_image": "data:image/png;base64,iVBORw0KGgo..."
}

# Raw base64 (auto-detected)
POST /api/sections/banner
Content-Type: application/json

{
  "banner_image": "iVBORw0KGgoAAAANSUhEUgAAAAE..."
}

# Multiple base64 images
POST /api/sections/portfolio
Content-Type: application/json

{
  "gallery_images": [
    "data:image/png;base64,iVBORw0KGgo...",
    "data:image/jpeg;base64,/9j/4AAQSkZ..."
  ]
}
```

#### Mixed Upload (File + Base64)
```bash
POST /api/sections/mixed
Content-Type: multipart/form-data

{
  "hero_image": [file],
  "about_image": "data:image/png;base64,iVBORw0KGgo...",
  "title": "Mixed Content"
}
```

### 3. **Automatic Full URL Return**
The model automatically converts stored paths to full URLs:
```json
{
  "id": 1,
  "name": "home",
  "body": {
    "title": "Home Section",
    "hero_image": "http://localhost/storage/sections/abc123.jpg",
    "hero_images": [
      "http://localhost/storage/sections/def456.jpg",
      "http://localhost/storage/sections/ghi789.jpg"
    ]
  }
}
```

### 4. **Automatic Image Management**
- **Old Images Deletion**: When updating a section with new images, old images are automatically deleted
- **Section Deletion**: When deleting a section, all associated images are removed from storage
- **Storage Location**: All images are stored in `storage/app/public/sections/`

## Supported Image Formats
- PNG
- JPEG/JPG
- GIF
- WebP

## API Endpoints

### Public Endpoints
- `GET /api/sections` - List all sections
- `GET /api/sections/{name}` - Get specific section by name

### Authenticated Endpoints (requires Sanctum token)
- `POST /api/sections/{name}` - Create or update section
- `DELETE /api/sections/{name}` - Delete section

## Example Usage

### 1. Upload Single Image with File
```bash
curl -X POST http://localhost/api/sections/home \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -F "title=Home Section" \
  -F "hero_image=@/path/to/image.jpg"
```

### 2. Upload Multiple Images
```bash
curl -X POST http://localhost/api/sections/gallery \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -F "title=Gallery" \
  -F "hero_images[]=@/path/to/image1.jpg" \
  -F "hero_images[]=@/path/to/image2.jpg"
```

### 3. Upload Base64 Image
```bash
curl -X POST http://localhost/api/sections/about \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "title": "About Us",
    "hero_image": "data:image/png;base64,iVBORw0KGgo..."
  }'
```

### 4. Mixed Data with Images
```bash
curl -X POST http://localhost/api/sections/contact \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -F "title=Contact Us" \
  -F "description=Get in touch" \
  -F "hero_image=@/path/to/image.jpg" \
  -F "count=5" \
  -F "tags[]=tag1" \
  -F "tags[]=tag2"
```

### 5. Retrieve Section (Public)
```bash
curl http://localhost/api/sections/home
```

Response:
```json
{
  "success": true,
  "data": {
    "id": 1,
    "name": "home",
    "body": {
      "title": "Home Section",
      "hero_image": "http://localhost/storage/sections/abc123.jpg"
    },
    "addedBy": {
      "id": 1,
      "name": "Admin"
    },
    "updatedBy": {
      "id": 1,
      "name": "Admin"
    }
  }
}
```

## Testing

All functionality is covered by comprehensive tests in `tests/Feature/SectionImageUploadTest.php`:

```bash
# Run the tests
php artisan test --filter=SectionImageUploadTest
```

Test Coverage:
- ✅ Single image upload with `_image` suffix
- ✅ Multiple images upload with `_images` suffix
- ✅ Various custom image field names
- ✅ Updating section replaces old images
- ✅ Deleting section removes all images
- ✅ Mixed data with images and regular fields
- ✅ Retrieving section returns full image URLs
- ✅ Base64 image upload with data URI
- ✅ Raw base64 image upload
- ✅ Multiple base64 images upload
- ✅ Mixed file and base64 upload

## Implementation Details

### Modified Files

1. **`app/Services/SectionService.php`**
   - Handles image upload processing (files and base64)
   - Manages old image deletion
   - Detects base64 images automatically

2. **`app/Models/Section.php`**
   - Added accessor for `body` attribute
   - Converts image paths to full URLs automatically

3. **`app/Http/Controllers/SectionController.php`**
   - Separates file uploads from regular data
   - Passes files to service layer

4. **`tests/Feature/SectionImageUploadTest.php`**
   - Comprehensive test suite for all features

## Storage Configuration

Make sure the storage is properly linked:
```bash
php artisan storage:link
```

This creates a symbolic link from `public/storage` to `storage/app/public`.

## Security Notes

- All image uploads require authentication (Sanctum token)
- Base64 detection validates image signatures to prevent malicious data
- Supported formats: PNG, JPEG, GIF, WebP only
- Old images are automatically cleaned up to prevent storage bloat





