# Flexible Data Implementation - Simplified Structure

## Overview

All modules now follow the same simple structure as Sections and Settings:
- **name** (string) - For identification
- **body** (JSON) - For ALL flexible data

This means dedicated fields like `logo`, `image`, `description`, `client_id`, `sector_id` have been removed. Everything goes in the flexible `body` field!

## Modules with Flexible Structure

### ✅ Sections (Original)
- Structure: `name` + `body`
- All data stored in flexible body field

### ✅ Settings (Original)
- Structure: `name` + `body`
- All data stored in flexible body field

### ✅ Clients (Updated to match)
- **Removed dedicated field**: `logo`
- Structure: `name` + `body`
- Everything stored in body (logo goes as `logo_image` in body)
- Migrations:
  - `2025_10_23_000001_add_body_to_clients_table.php` - Added body field
  - `2025_10_23_000010_simplify_clients_to_flexible_structure.php` - Removed logo field

### ✅ Sectors (Updated to match)
- **Removed dedicated field**: `image`
- Structure: `name` + `body`
- Everything stored in body (image goes as `hero_image` or similar in body)
- Migrations:
  - `2025_10_23_000002_add_body_to_sectors_table.php` - Added body field
  - `2025_10_23_000011_simplify_sectors_to_flexible_structure.php` - Removed image field

### ✅ Projects (Updated to match)
- **Removed dedicated fields**: `description`, `image`, `client_id`, `sector_id`
- Structure: `name` + `body`
- Everything stored in body (relationships stored as text/data in body)
- Migrations:
  - `2025_10_23_000003_add_body_to_projects_table.php` - Added body field
  - `2025_10_23_000012_simplify_projects_to_flexible_structure.php` - Removed dedicated fields

## Modules WITHOUT Flexible Body Data

### ❌ Users
- Remains with standard fields only (name, email, password)
- No flexible body data

### ❌ Mails
- Remains with standard fields only (name, email, subject, body, status, etc.)
- No flexible body data

## Features

### 1. Dynamic Data Storage
Each module can now store any arbitrary data in the `body` JSON field:

```json
{
  "custom_field": "value",
  "another_field": "another value",
  "hero_image": "path/to/image.jpg",
  "gallery_images": ["path/1.jpg", "path/2.jpg"]
}
```

### 2. Automatic Image Handling

The system automatically handles images in the body field:

- **Single images**: Any field ending with `_image` (e.g., `hero_image`, `banner_image`)
- **Multiple images**: Any field ending with `_images` (e.g., `gallery_images`, `slider_images`)

Both file uploads and base64 encoded images are supported.

### 3. URL Transformation

When retrieving data, image paths are automatically converted to full URLs:
- `sectors/image.jpg` → `http://yoursite.com/storage/sectors/image.jpg`

### 4. Automatic Cleanup

When updating or deleting records, old images are automatically removed from storage.

## Implementation Details

### FlexibleBodyTrait

A new trait `app/Services/FlexibleBodyTrait.php` provides shared functionality:

- `processImageUploads()` - Process both file uploads and base64 images
- `getRawBodyData()` - Get raw JSON data without transformation
- `deleteAllImages()` - Clean up images when deleting records
- `isBase64Image()` - Detect base64 encoded images
- `storeBase64Image()` - Store base64 images to disk
- `flattenFiles()` - Handle nested file arrays from requests

### Models Updated

All models with flexible body support now include:
- `body` field in `$fillable` array
- `body` cast to `array` in `$casts`
- `body()` Attribute accessor for URL transformation

### Services Updated

All services now simplified to match SectionService:
- `listAll()` - Get all records
- `findById(int $id)` - Find by ID
- `findByName(string $name)` - Find by name
- `upsertByName(string $name, array $bodyData, array $files)` - Create or update
- `deleteByName(string $name)` - Delete by name

No more separate parameters for dedicated fields - everything goes in `bodyData`!

## Migrations

Run the following migrations to update the database structure:

```bash
php artisan migrate
```

This will execute (in order):
1. `2025_10_23_000001_add_body_to_clients_table.php` - Add body field
2. `2025_10_23_000002_add_body_to_sectors_table.php` - Add body field
3. `2025_10_23_000003_add_body_to_projects_table.php` - Add body field
4. `2025_10_23_000010_simplify_clients_to_flexible_structure.php` - Remove logo field
5. `2025_10_23_000011_simplify_sectors_to_flexible_structure.php` - Remove image field
6. `2025_10_23_000012_simplify_projects_to_flexible_structure.php` - Remove description, image, client_id, sector_id fields

## Usage Examples

### Example 1: Client with All Data in Body

```http
POST /api/clients/acme-corp
Content-Type: multipart/form-data

{
  "website": "https://acme.com",
  "industry": "Technology",
  "contact_person": "John Doe",
  "phone": "+1-555-0123",
  "logo_image": <file or base64>
}
```

### Example 2: Project with Everything in Body

```http
POST /api/projects/new-building
Content-Type: application/json

{
  "description": "Modern office complex",
  "client": "Acme Corp",
  "sector": "Commercial",
  "location": "Downtown",
  "budget": "5M USD",
  "completion_date": "2025-12-31",
  "gallery_images": [<files>]
}
```

### Example 3: Sector with Flexible Data

```http
POST /api/sectors/healthcare
Content-Type: application/json

{
  "description": "Healthcare sector projects",
  "color": "#2E7D32",
  "key_features": ["Modern", "Sustainable"],
  "hero_image": <file or base64>
}
```

## Benefits

1. **Ultimate Flexibility**: Add any custom fields without database schema changes
2. **Consistent Structure**: All modules work the same way (name + body)
3. **Simplified API**: Same endpoints and patterns across all modules
4. **Image Management**: Automatic handling of image uploads and cleanup
5. **URL Transformation**: Automatic conversion of storage paths to full URLs
6. **No Relationships in DB**: Store client/sector names as text instead of foreign keys
7. **Clean Code**: Shared functionality through trait reduces duplication
8. **Easy to Understand**: Only two fields per module - name and body!

## Important Notes

- **No dedicated fields anymore**: Everything except `name` goes in `body`
- **Image fields** must end with `_image` (singular) or `_images` (plural) to be processed automatically
- **All images** are stored in their respective module directories:
  - `storage/app/public/clients/`
  - `storage/app/public/sectors/`
  - `storage/app/public/projects/`
  - `storage/app/public/sections/`
  - `storage/app/public/settings/`
- **Base64 images** are automatically detected and converted to files
- **Relationships**: Projects can reference clients/sectors by name in the body (not database foreign keys)
- **Complete freedom**: Store any JSON structure you want - objects, arrays, nested data, etc.

## API Endpoints

All modules now use the same simple endpoints:

- `GET /api/{module}` - List all
- `GET /api/{module}/{name}` - Get by name
- `POST /api/{module}/{name}` - Create or update (upsert)
- `DELETE /api/{module}/{name}` - Delete by name

Where `{module}` is one of: `clients`, `sectors`, `projects`, `sections`, `settings`

